import { ChangeDetectorRef, Component, OnInit, ViewChild } from '@angular/core';
import { MatDialog } from '@angular/material/dialog';
import { MatPaginator, PageEvent } from '@angular/material/paginator';
import { MatSort } from '@angular/material/sort';
import { MatTableDataSource } from '@angular/material/table';
import { BercosService } from '../../../services/bercos/bercos.service';
import { CoreService } from '../../../core/core.service';
import { CommonModule } from '@angular/common';
import { FormsModule, ReactiveFormsModule } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatInputModule } from '@angular/material/input';
import { MatFormFieldModule } from '@angular/material/form-field';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatSelectModule } from '@angular/material/select';
import { MatRadioModule } from '@angular/material/radio';
import { MatIconModule } from '@angular/material/icon';
import { MatSnackBarModule } from '@angular/material/snack-bar';
import { MatTableModule } from '@angular/material/table';
import { MatToolbarModule } from '@angular/material/toolbar';
import { MatPaginatorModule } from '@angular/material/paginator';
import { MatSortModule } from '@angular/material/sort';
import { BercoAddComponent } from '../berco-add/berco-add.component';
import { BercoEditComponent } from '../berco-edit/berco-edit.component';
import { VisualizacaoBercosComponent } from '../visualizacao-bercos/visualizacao-bercos.component';
import { AddSearchComponent } from '../../../shared/addAndSearch/add-search/add-search.component';
import { BercoSearchComponent } from '../berco-search/berco-search.component';
import { JwtDecoderService } from '../../../jwt-decoder.service';
import { AuthService } from '../../../auth.service';
import { jwtDecode } from 'jwt-decode';
import { CsvService } from '../../../csv.service';

interface BercoData {
  id: number;
  nome: string;
  compri_estrutural: number;
  compri_util: number;
  dwt: number;
  largura: number;
  profundidade: number;
  calado_max: number;
  boca_max: number;
  loa_max: number;
  action: number;
}

@Component({
  selector: 'app-bercos',
  standalone: true,
  imports: [
    MatPaginatorModule,
    MatToolbarModule,
    MatTableModule,
    CommonModule,
    FormsModule,
    ReactiveFormsModule,
    MatButtonModule,
    MatInputModule,
    MatFormFieldModule,
    MatDatepickerModule,
    MatSelectModule,
    MatRadioModule,
    MatIconModule,
    MatSnackBarModule,
    MatSortModule,
    AddSearchComponent,
  ],
  templateUrl: './bercos.component.html',
  styleUrls: ['./bercos.component.scss'],
})
export class BercosComponent implements OnInit {
  totalItems = 100;
  pageSize = 10;
  currentPage = 1;

  displayedColumns: string[] = [
    'id',
    'nome',
    'compri_estrutural',
    'compri_util',
    'dwt',
    'largura',
    'profundidade',
    'action',
  ];

  dataSource = new MatTableDataSource<BercoData>();

  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;
  user: any;
  authTokenPresent = false;

  constructor(
    private _dialog: MatDialog,
    private _bercoService: BercosService,
    private _coreService: CoreService,
    private jwtDecoderService: JwtDecoderService,
    private authService: AuthService,
    private cdr: ChangeDetectorRef,
    private _csvService: CsvService
  ) { }

  ngOnInit(): void {
    this.validationToken();
  }

  ngAfterViewInit(): void {
    this.loadData();
  }
  validationToken() {
    this.authService.authToken$.subscribe((isLoggedIn) => {
      this.authTokenPresent = isLoggedIn;
      if (this.authTokenPresent) {
        const token = sessionStorage.getItem('accessToken')!;
        this.user = jwtDecode(token);
        const isExpired =
          this.user && this.user.exp
            ? this.user.exp < Date.now() / 1000
            : false;
        if (isExpired) {
          sessionStorage.removeItem('accessToken');
        }
        this.user = this.jwtDecoderService.decodeToken(token);
        console.log(this.user);
      }
      this.cdr.detectChanges();
    });
  }

  loadData(event?: PageEvent): void {
    if (typeof sessionStorage !== 'undefined') {
      const storedData = sessionStorage.getItem('bercosSearch');
      if (storedData) {
        const bercos = JSON.parse(storedData); // Converte o JSON em um objeto
        console.log('Dados recuperados do sessionStorage:', bercos);

        // Atualiza os dados no MatTableDataSource
        this.dataSource.data = bercos;

        // Reassocia paginator e sort
        if (this.paginator && this.sort) {
          this.dataSource.paginator = this.paginator;
          this.dataSource.sort = this.sort;
        }

        // Atualiza a visualização
        this.cdr.detectChanges();

        // Remove dados do sessionStorage após o uso
        sessionStorage.removeItem('bercosSearch');
        console.log('bercosSearch removido do sessionStorage.');

        return; // Retorna aqui, pois já usamos os dados do sessionStorage
      } else {
        console.warn('Nenhum dado encontrado no sessionStorage.');
      }
    }

    // Caso os dados não estejam no sessionStorage, chama a API
    const pageIndex = event ? event.pageIndex + 1 : this.currentPage - 1;

    this._bercoService.getEmployeeList(pageIndex, this.pageSize).subscribe({
      next: (res: any) => {
        const bercos = res._embedded.bercoResponseList;
        console.log(bercos);
        this.dataSource.data = bercos;
        this.totalItems = res.page.totalElements;
        console.log(res.page.totalElements);
      },
      error: (err: any) => {
        console.error(err);
        if (err.error.title) {
          this._coreService.openSnackBar(err.error.title);
        } else {
          this._coreService.openSnackBar(err.error.message);
        }
      },
    });
  }


  applyFilter(event: Event) {
    const filterValue = (event.target as HTMLInputElement).value;
    this.dataSource.filter = filterValue.trim().toLowerCase();

    if (this.dataSource.paginator) {
      this.dataSource.paginator.firstPage();
    }
  }

  deleteBerco(id: number) {
    this._bercoService.deleteEmployee(id).subscribe({
      next: () => {
        this._coreService.openSnackBar('Berço deleted!', 'done');
        this.loadData();
      },
      error: (err: any) => {
        this._coreService.openSnackBar(err.error.title);
      },
    });
    this.cdr.detectChanges();

  }

  openEditForm(data: any) {
    this._dialog.open(BercoEditComponent, {
      data,
      autoFocus: true, // Foca no primeiro campo interativo do formulário
      width: '600px', // Define a largura do diálogo
    });
    this.cdr.detectChanges();
  }

  openAddBercoForm() {
    this._dialog.open(BercoAddComponent, {
      autoFocus: true, // Foca no primeiro campo interativo do formulário
      disableClose: true, // Impede que o diálogo feche ao clicar fora
      width: '500px', // Define a largura do diálogo});
    });
    this.cdr.detectChanges();
  }

  openSearchBercoForm() {
    this._dialog.open(BercoSearchComponent);
    if (typeof sessionStorage !== 'undefined') {
      const storedData = sessionStorage.getItem('bercosSearch');
      if (storedData) {
        const dataSource = JSON.parse(storedData); // Converte o JSON em um objeto
        console.log('Dados recuperados do sessionStorage:', dataSource);
        this.dataSource = dataSource;
        this.cdr.detectChanges();
        // Atualiza a tabela com os dados
      } else {
        console.warn('Nenhum dado encontrado no sessionStorage.');
      }
    } else {
      console.error('sessionStorage não está disponível.');
    }
  }

  getBercoById(data: any) {
    this._dialog.open(VisualizacaoBercosComponent, {
      data,
      autoFocus: true, // Foca no primeiro campo interativo do formulário
      width: '800px', // Define a largura do diálogo
    });
  }

  public saveDataInCSV(name: string): void {
    this._bercoService.getAllAceitesSemPaginacao().subscribe({
      next: (aceites: any[]) => {
        console.log('✅ Aceites recebidos:', aceites); // LOGA TODOS OS ACEITES ORIGINAIS

        const data = aceites.map(a => {
          const userRaw = a.user || '';
          let idUsuario = '';
          let email = '';

          console.log('🔍 Usuário bruto:', userRaw); // LOGA STRING DO USUÁRIO

          // Extrai ID e email do campo "Usuário"
          const match = userRaw.match(/User\(id=(\d+),\s*email=([^)]+)\)/);
          if (match) {
            idUsuario = match[1];
            email = match[2];
          } else if (userRaw.match(/User\(id=(\d+)\)/)) {
            idUsuario = userRaw.match(/User\(id=(\d+)\)/)[1];
          }

          // Categoria
          let categoriaLabel = 'Não informado';
          switch (+a.categoria) {
            case 1:
              categoriaLabel = 'Granel Sólido';
              break;
            case 2:
              categoriaLabel = 'Granel Líquido';
              break;
            case 3:
              categoriaLabel = 'Carga Geral';
              break;
          }


          const row = {
            Nome: a.nome,
            DWT: a.dwt || '',
            'Boca Máx': a.boca_max || '',
            'Calado Máx': a.calado_max || '',
            'Categoria': categoriaLabel,
            Profundidade: a.profundidade || '',
            Largura: a.largura || '',
            'LOA Máx': a.loa_max || '',
            'Comprimento Estrutural': a.compri_estrutural || '',
            'Comprimento Útil': a.compri_util || '',
          };


          return row;
        });

        console.log('📋 Dados finais para CSV:', data); // LOGA O ARRAY COMPLETO FINAL

        const csvContent = this._csvService.saveDataInCSV(data);
        console.log('📝 Conteúdo CSV:', csvContent); // LOGA O TEXTO CSV GERADO

        const hiddenElement = document.createElement('a');
        hiddenElement.href = 'data:text/csv;charset=utf-8,' + encodeURI(csvContent);
        hiddenElement.target = '_blank';
        hiddenElement.download = name + '.csv';
        hiddenElement.click();
      },
      error: (err) => {
        this._coreService.openSnackBar('Erro ao exportar CSV');
        console.error('❌ Erro ao exportar CSV:', err);
      }
    });
  }
}
