import { ChangeDetectorRef, Component } from '@angular/core';
import { DefaultLoginLayoutComponent } from '../../components/default-login-layout/default-login-layout.component';
import {
  FormControl,
  FormGroup,
  FormRecord,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { PrimaryInputComponent } from '../../components/primary-input/primary-input.component';
import { Router } from '@angular/router';
import { LoginService } from '../../services/login.service';
import { ToastrService } from 'ngx-toastr';
import { AuthService } from '../../auth.service';
import { JwtDecoderService } from '../../jwt-decoder.service';
interface LoginForm {
  email: FormControl;
  password: FormControl;
}

@Component({
  selector: 'app-login',
  standalone: true,
  imports: [
    DefaultLoginLayoutComponent,
    ReactiveFormsModule,
    PrimaryInputComponent,
  ],
  providers: [LoginService],
  templateUrl: './login.component.html',
  styleUrl: './login.component.scss',
})
export class LoginComponent {
  loginForm!: FormGroup<LoginForm>;

  constructor(
    private router: Router,
    private authService: AuthService,
    private loginService: LoginService,
    private toastService: ToastrService,
    private jwtDecoderService: JwtDecoderService
  ) {
    this.loginForm = new FormGroup({
      email: new FormControl('', [Validators.required, Validators.email]),
      password: new FormControl('', [
        Validators.required,
        Validators.minLength(6),
      ]),
    });
  }

  submit() {
  this.loginService
    .login(this.loginForm.value.email, this.loginForm.value.password)
    .subscribe({
      next: (response) => {
        this.authService.updateAuthTokenStatus(true);
        const decodedToken = this.jwtDecoderService.decodeToken(response.accessToken);
        if (decodedToken.role === 'COMPANY') {
          this.router.navigate(['/dashboard']);
        } else {
          this.router.navigate(['/navio']);
        }
        this.toastService.success('Login feito com sucesso!');
      },
      error: () =>
        this.toastService.error('Erro inesperado! Tente novamente mais tarde'),
    });
}

  navigate() {
    this.router.navigate(['/signup']);
  }
}
